CREATE TABLE IF NOT EXISTS users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    email VARCHAR(190) NOT NULL UNIQUE,
    password_hash VARCHAR(255) NOT NULL,
    name VARCHAR(190),
    role ENUM('user', 'admin') DEFAULT 'user',
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS articles (
    id INT AUTO_INCREMENT PRIMARY KEY,
    slug VARCHAR(190) NOT NULL UNIQUE,
    title VARCHAR(190) NOT NULL,
    excerpt TEXT,
    body LONGTEXT NOT NULL,
    category VARCHAR(100),
    published TINYINT(1) DEFAULT 1,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS quotes (
    id INT AUTO_INCREMENT PRIMARY KEY,
    text TEXT NOT NULL,
    author VARCHAR(190),
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS user_stats (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    total_eye_exercises_minutes INT DEFAULT 0,
    total_focus_minutes INT DEFAULT 0,
    total_breathing_sessions INT DEFAULT 0,
    last_activity_at DATETIME,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Phase 2 Tables

CREATE TABLE IF NOT EXISTS challenges (
    id INT AUTO_INCREMENT PRIMARY KEY,
    title VARCHAR(190) NOT NULL,
    description TEXT,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS user_challenges (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    challenge_id INT NOT NULL,
    date_assigned DATE NOT NULL,
    completed TINYINT(1) DEFAULT 0,
    completed_at DATETIME,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (challenge_id) REFERENCES challenges(id) ON DELETE CASCADE,
    UNIQUE KEY unique_daily_challenge (user_id, date_assigned)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS daily_mood (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    mood_value INT NOT NULL, -- 1 to 5
    note TEXT,
    date DATE NOT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_daily_mood (user_id, date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Phase 3 Tables

CREATE TABLE IF NOT EXISTS learning_paths (
    id INT AUTO_INCREMENT PRIMARY KEY,
    slug VARCHAR(190) NOT NULL UNIQUE,
    title VARCHAR(190) NOT NULL,
    short_description TEXT,
    days INT NOT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS learning_steps (
    id INT AUTO_INCREMENT PRIMARY KEY,
    path_id INT NOT NULL,
    day_number INT NOT NULL,
    title VARCHAR(190) NOT NULL,
    content LONGTEXT NOT NULL,
    suggested_tools VARCHAR(255),
    FOREIGN KEY (path_id) REFERENCES learning_paths(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Seed Data (Insert Ignore to prevent duplicates on re-run)

-- Learning Paths
INSERT IGNORE INTO learning_paths (id, slug, title, short_description, days) VALUES
(1, 'fix-screen-fatigue', 'Fix Screen Fatigue in 5 Days', 'A 5-day guide to reducing eye strain and improving digital wellness.', 5),
(2, 'boost-focus', 'Boost Your Focus in 7 Days', 'Master deep work and eliminate distractions in one week.', 7),
(3, 'calm-mind', 'Calm Your Mind in 3 Days', 'Quick techniques to reduce anxiety and find balance.', 3);

-- Learning Steps (Sample for Path 1)
INSERT IGNORE INTO learning_steps (path_id, day_number, title, content, suggested_tools) VALUES
(1, 1, 'Understanding Eye Strain', 'Learn why your eyes get tired and the 20-20-20 rule.', 'eye_exercises'),
(1, 2, 'Lighting Matters', 'Adjust your screen brightness and ambient lighting.', 'eye_exercises'),
(1, 3, 'Blink More', 'We blink less when looking at screens. Practice conscious blinking.', 'eye_exercises'),
(1, 4, 'Distance Focus', 'Take breaks to look at distant objects.', 'eye_exercises'),
(1, 5, 'Daily Routine', 'Build a sustainable eye care habit.', 'eye_exercises');

-- Articles
INSERT IGNORE INTO articles (slug, title, excerpt, body, category, published) VALUES
('eye-strain-basics', 'Eye Strain Basics', 'Why your eyes hurt and what to do.', 'Digital eye strain is common. Symptoms include dry eyes, headaches, and blurred vision. The best cure is taking regular breaks.', 'Health', 1),
('deep-work-intro', 'Introduction to Deep Work', 'Focus is a superpower.', 'Deep work is the ability to focus without distraction on a cognitively demanding task. It is a skill that allows you to quickly master complicated information and produce better results in less time.', 'Productivity', 1),
('breathing-101', 'Breathing 101', 'Simple techniques for calm.', 'Box breathing is a simple technique used by Navy SEALs to stay calm and focused in tense situations.', 'Wellness', 1),
('sleep-hygiene', 'Sleep Hygiene Tips', 'Better sleep for better days.', 'Keep your room cool and dark. Avoid screens 1 hour before bed. Stick to a schedule.', 'Health', 1),
('digital-detox', 'Digital Detox Guide', 'Unplug to recharge.', 'Taking a break from technology can reduce stress and improve mental clarity. Try a 24-hour detox this weekend.', 'Wellness', 1),
('productivity-hacks', 'Top Productivity Hacks', 'Get more done in less time.', 'Use the Pomodoro technique. Eat the frog (do the hardest task first). Batch similar tasks.', 'Productivity', 1),
('morning-routine', 'Perfect Morning Routine', 'Start your day right.', 'Hydrate first thing. Move your body. Plan your day. Avoid checking email immediately.', 'Lifestyle', 1),
('mindfulness-work', 'Mindfulness at Work', 'Stay present.', 'Take mindful breaths before meetings. Focus on one task at a time. Listen actively.', 'Wellness', 1),
('ergonomics', 'Office Ergonomics', 'Sit better, feel better.', 'Adjust your chair height so your feet are flat on the floor. Keep your monitor at eye level.', 'Health', 1),
('hydration', 'Importance of Hydration', 'Drink up.', 'Water is essential for brain function. Dehydration can lead to fatigue and headaches.', 'Health', 1);

-- Quotes
INSERT IGNORE INTO quotes (text, author) VALUES
('The only way to do great work is to love what you do.', 'Steve Jobs'),
('Focus on being productive instead of busy.', 'Tim Ferriss'),
('Your mind will answer most questions if you learn to relax and wait for the answer.', 'William S. Burroughs'),
('Almost everything will work again if you unplug it for a few minutes, including you.', 'Anne Lamott'),
('Rest is not idleness.', 'John Lubbock'),
('Tension is who you think you should be. Relaxation is who you are.', 'Chinese Proverb'),
('The time to relax is when you don\'t have time for it.', 'Sydney J. Harris'),
('Breathe. Let go. And remind yourself that this very moment is the only one you know you have for sure.', 'Oprah Winfrey'),
('It is not a daily increase, but a daily decrease. Hack away at the unessential.', 'Bruce Lee'),
('Simplicity is the ultimate sophistication.', 'Leonardo da Vinci');

-- Challenges (Additional to seed in code)
INSERT IGNORE INTO challenges (title, description) VALUES
('20-20-20 Rule', 'Every 20 minutes, look at something 20 feet away for 20 seconds.'),
('Focus Session', 'Complete one 25-minute focus session without interruptions.'),
('Hydration Goal', 'Drink 2 liters of water today.'),
('No Screens Lunch', 'Eat your lunch away from your computer and phone.'),
('Evening Walk', 'Take a 20-minute walk after dinner.'),
('Sleep Early', 'Go to bed 30 minutes earlier than usual.'),
('Declutter', 'Clean your desk or workspace for 10 minutes.'),
('Kindness', 'Send a kind message to a friend or colleague.'),
('Stretch Break', 'Do a 5-minute stretching routine.'),
('Journaling', 'Write down 3 thoughts or feelings in a journal.');
