-- Users Table
CREATE TABLE IF NOT EXISTS users (
    id SERIAL PRIMARY KEY,
    email VARCHAR(190) NOT NULL UNIQUE,
    password_hash VARCHAR(255) NOT NULL,
    name VARCHAR(190),
    role VARCHAR(20) DEFAULT 'user' CHECK (role IN ('user', 'admin')),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Articles Table
CREATE TABLE IF NOT EXISTS articles (
    id SERIAL PRIMARY KEY,
    slug VARCHAR(190) NOT NULL UNIQUE,
    title VARCHAR(190) NOT NULL,
    excerpt TEXT,
    body TEXT NOT NULL,
    category VARCHAR(100),
    published BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Quotes Table
CREATE TABLE IF NOT EXISTS quotes (
    id SERIAL PRIMARY KEY,
    text TEXT NOT NULL,
    author VARCHAR(190),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- User Stats Table
CREATE TABLE IF NOT EXISTS user_stats (
    id SERIAL PRIMARY KEY,
    user_id INTEGER NOT NULL REFERENCES users(id) ON DELETE CASCADE,
    total_eye_exercises_minutes INTEGER DEFAULT 0,
    total_focus_minutes INTEGER DEFAULT 0,
    total_breathing_sessions INTEGER DEFAULT 0,
    last_activity_at TIMESTAMP
);

-- Challenges Table
CREATE TABLE IF NOT EXISTS challenges (
    id SERIAL PRIMARY KEY,
    title VARCHAR(190) NOT NULL,
    description TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- User Challenges Table
CREATE TABLE IF NOT EXISTS user_challenges (
    id SERIAL PRIMARY KEY,
    user_id INTEGER NOT NULL REFERENCES users(id) ON DELETE CASCADE,
    challenge_id INTEGER NOT NULL REFERENCES challenges(id) ON DELETE CASCADE,
    date_assigned DATE NOT NULL,
    completed BOOLEAN DEFAULT FALSE,
    completed_at TIMESTAMP,
    UNIQUE (user_id, date_assigned)
);

-- Daily Mood Table
CREATE TABLE IF NOT EXISTS daily_mood (
    id SERIAL PRIMARY KEY,
    user_id INTEGER NOT NULL REFERENCES users(id) ON DELETE CASCADE,
    mood_value INTEGER NOT NULL, -- 1 to 5
    note TEXT,
    date DATE NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE (user_id, date)
);

-- Learning Paths Table
CREATE TABLE IF NOT EXISTS learning_paths (
    id SERIAL PRIMARY KEY,
    slug VARCHAR(190) NOT NULL UNIQUE,
    title VARCHAR(190) NOT NULL,
    short_description TEXT,
    days INTEGER NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

-- Learning Steps Table
CREATE TABLE IF NOT EXISTS learning_steps (
    id SERIAL PRIMARY KEY,
    path_id INTEGER NOT NULL REFERENCES learning_paths(id) ON DELETE CASCADE,
    day_number INTEGER NOT NULL,
    title VARCHAR(190) NOT NULL,
    content TEXT NOT NULL,
    suggested_tools VARCHAR(255)
);

-- Seed Data

-- Learning Paths
INSERT INTO learning_paths (id, slug, title, short_description, days) VALUES
(1, 'fix-screen-fatigue', 'Fix Screen Fatigue in 5 Days', 'A 5-day guide to reducing eye strain and improving digital wellness.', 5),
(2, 'boost-focus', 'Boost Your Focus in 7 Days', 'Master deep work and eliminate distractions in one week.', 7),
(3, 'calm-mind', 'Calm Your Mind in 3 Days', 'Quick techniques to reduce anxiety and find balance.', 3)
ON CONFLICT (id) DO NOTHING;

-- Learning Steps (Sample for Path 1)
INSERT INTO learning_steps (path_id, day_number, title, content, suggested_tools) VALUES
(1, 1, 'Understanding Eye Strain', 'Learn why your eyes get tired and the 20-20-20 rule.', 'eye_exercises'),
(1, 2, 'Lighting Matters', 'Adjust your screen brightness and ambient lighting.', 'eye_exercises'),
(1, 3, 'Blink More', 'We blink less when looking at screens. Practice conscious blinking.', 'eye_exercises'),
(1, 4, 'Distance Focus', 'Take breaks to look at distant objects.', 'eye_exercises'),
(1, 5, 'Daily Routine', 'Build a sustainable eye care habit.', 'eye_exercises')
ON CONFLICT DO NOTHING;

-- Articles
INSERT INTO articles (slug, title, excerpt, body, category, published) VALUES
('eye-strain-basics', 'Eye Strain Basics', 'Why your eyes hurt and what to do.', 'Digital eye strain is common. Symptoms include dry eyes, headaches, and blurred vision. The best cure is taking regular breaks.', 'Health', TRUE),
('deep-work-intro', 'Introduction to Deep Work', 'Focus is a superpower.', 'Deep work is the ability to focus without distraction on a cognitively demanding task. It is a skill that allows you to quickly master complicated information and produce better results in less time.', 'Productivity', TRUE),
('breathing-101', 'Breathing 101', 'Simple techniques for calm.', 'Box breathing is a simple technique used by Navy SEALs to stay calm and focused in tense situations.', 'Wellness', TRUE),
('sleep-hygiene', 'Sleep Hygiene Tips', 'Better sleep for better days.', 'Keep your room cool and dark. Avoid screens 1 hour before bed. Stick to a schedule.', 'Health', TRUE),
('digital-detox', 'Digital Detox Guide', 'Unplug to recharge.', 'Taking a break from technology can reduce stress and improve mental clarity. Try a 24-hour detox this weekend.', 'Wellness', TRUE),
('productivity-hacks', 'Top Productivity Hacks', 'Get more done in less time.', 'Use the Pomodoro technique. Eat the frog (do the hardest task first). Batch similar tasks.', 'Productivity', TRUE),
('morning-routine', 'Perfect Morning Routine', 'Start your day right.', 'Hydrate first thing. Move your body. Plan your day. Avoid checking email immediately.', 'Lifestyle', TRUE),
('mindfulness-work', 'Mindfulness at Work', 'Stay present.', 'Take mindful breaths before meetings. Focus on one task at a time. Listen actively.', 'Wellness', TRUE),
('ergonomics', 'Office Ergonomics', 'Sit better, feel better.', 'Adjust your chair height so your feet are flat on the floor. Keep your monitor at eye level.', 'Health', TRUE),
('hydration', 'Importance of Hydration', 'Drink up.', 'Water is essential for brain function. Dehydration can lead to fatigue and headaches.', 'Health', TRUE)
ON CONFLICT (slug) DO NOTHING;

-- Quotes
INSERT INTO quotes (text, author) VALUES
('The only way to do great work is to love what you do.', 'Steve Jobs'),
('Focus on being productive instead of busy.', 'Tim Ferriss'),
('Your mind will answer most questions if you learn to relax and wait for the answer.', 'William S. Burroughs'),
('Almost everything will work again if you unplug it for a few minutes, including you.', 'Anne Lamott'),
('Rest is not idleness.', 'John Lubbock'),
('Tension is who you think you should be. Relaxation is who you are.', 'Chinese Proverb'),
('The time to relax is when you don''t have time for it.', 'Sydney J. Harris'),
('Breathe. Let go. And remind yourself that this very moment is the only one you know you have for sure.', 'Oprah Winfrey'),
('It is not a daily increase, but a daily decrease. Hack away at the unessential.', 'Bruce Lee'),
('Simplicity is the ultimate sophistication.', 'Leonardo da Vinci')
ON CONFLICT DO NOTHING;

-- Challenges
INSERT INTO challenges (title, description) VALUES
('20-20-20 Rule', 'Every 20 minutes, look at something 20 feet away for 20 seconds.'),
('Focus Session', 'Complete one 25-minute focus session without interruptions.'),
('Hydration Goal', 'Drink 2 liters of water today.'),
('No Screens Lunch', 'Eat your lunch away from your computer and phone.'),
('Evening Walk', 'Take a 20-minute walk after dinner.'),
('Sleep Early', 'Go to bed 30 minutes earlier than usual.'),
('Declutter', 'Clean your desk or workspace for 10 minutes.'),
('Kindness', 'Send a kind message to a friend or colleague.'),
('Stretch Break', 'Do a 5-minute stretching routine.'),
('Journaling', 'Write down 3 thoughts or feelings in a journal.')
ON CONFLICT DO NOTHING;
