<?php

/**
 * Eye Routine Controller
 * Handles CRUD operations for eye routines and exercise assignments
 */

class EyeRoutineController {
    private $pdo;
    
    public function __construct($pdo) {
        $this->pdo = $pdo;
    }
    
    /**
     * Get all routines with optional filtering
     * 
     * @param array $filters ['difficulty', 'is_active', 'is_ai_generated', 'search']
     * @return array
     */
    public function getAllRoutines($filters = []) {
        $sql = "SELECT r.*, 
                COUNT(DISTINCT re.id) as exercise_count
                FROM eye_routines r
                LEFT JOIN eye_routine_exercises re ON r.id = re.routine_id
                WHERE 1=1";
        $params = [];
        
        if (isset($filters['difficulty'])) {
            $sql .= " AND r.difficulty = :difficulty";
            $params['difficulty'] = $filters['difficulty'];
        }
        
        if (isset($filters['is_active'])) {
            $sql .= " AND r.is_active = :is_active";
            $params['is_active'] = $filters['is_active'];
        }
        
        if (isset($filters['is_ai_generated'])) {
            $sql .= " AND r.is_ai_generated = :is_ai_generated";
            $params['is_ai_generated'] = $filters['is_ai_generated'];
        }
        
        if (!empty($filters['search'])) {
            $sql .= " AND (r.title LIKE :search OR r.slug LIKE :search)";
            $params['search'] = '%' . $filters['search'] . '%';
        }
        
        $sql .= " GROUP BY r.id ORDER BY r.created_at DESC";
        
        try {
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            error_log("Error fetching routines: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get routine by ID with optional exercise details
     * 
     * @param int $id
     * @param bool $includeExercises
     * @return array|null
     */
    public function getRoutineById($id, $includeExercises = false) {
        try {
            $stmt = $this->pdo->prepare("SELECT * FROM eye_routines WHERE id = :id");
            $stmt->execute(['id' => $id]);
            $routine = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$routine) {
                return null;
            }
            
            if ($includeExercises) {
                $routine['exercises'] = $this->getRoutineExercises($id);
            }
            
            return $routine;
        } catch (Exception $e) {
            error_log("Error fetching routine by ID: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Get all exercises assigned to a routine with details
     * 
     * @param int $routineId
     * @return array
     */
    public function getRoutineExercises($routineId) {
        try {
            $stmt = $this->pdo->prepare("
                SELECT 
                    re.id as routine_exercise_id,
                    re.exercise_id,
                    re.sort_order,
                    re.repetitions,
                    re.override_duration_seconds,
                    e.title as exercise_title,
                    e.slug as exercise_slug,
                    e.short_description as exercise_description,
                    e.difficulty as exercise_difficulty,
                    e.category as exercise_category,
                    e.default_duration_seconds as exercise_default_duration,
                    e.is_active as exercise_is_active,
                    COALESCE(re.override_duration_seconds, e.default_duration_seconds) as effective_duration,
                    (re.repetitions * COALESCE(re.override_duration_seconds, e.default_duration_seconds)) as total_contribution
                FROM eye_routine_exercises re
                JOIN eye_exercises e ON re.exercise_id = e.id
                WHERE re.routine_id = :routine_id
                ORDER BY re.sort_order ASC
            ");
            $stmt->execute(['routine_id' => $routineId]);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            error_log("Error fetching routine exercises: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Create new routine
     * 
     * @param array $data Routine metadata
     * @return int|false Routine ID or false
     */
    public function createRoutine($data) {
        // Validate data
        $errors = $this->validateRoutineData($data);
        if (!empty($errors)) {
            error_log("Routine validation failed: " . implode(', ', $errors));
            return false;
        }
        
        // Generate slug if not provided
        if (empty($data['slug'])) {
            $data['slug'] = $this->generateSlug($data['title']);
        } else {
            $data['slug'] = $this->generateSlug($data['slug']);
        }
        
        try {
            $stmt = $this->pdo->prepare("
                INSERT INTO eye_routines (
                    slug, title, description, difficulty,
                    total_duration_seconds, is_ai_generated, is_active,
                    created_at, updated_at
                ) VALUES (
                    :slug, :title, :description, :difficulty,
                    0, :is_ai_generated, :is_active,
                    NOW(), NOW()
                )
            ");
            
            $stmt->execute([
                'slug' => $data['slug'],
                'title' => $data['title'],
                'description' => $data['description'] ?? null,
                'difficulty' => $data['difficulty'] ?? 'easy',
                'is_ai_generated' => $data['is_ai_generated'] ?? 0,
                'is_active' => $data['is_active'] ?? 1
            ]);
            
            return $this->pdo->lastInsertId();
        } catch (Exception $e) {
            error_log("Error creating routine: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Update existing routine
     * 
     * @param int $id
     * @param array $data Fields to update
     * @return bool
     */
    public function updateRoutine($id, $data) {
        // Validate data
        $errors = $this->validateRoutineData($data, $id);
        if (!empty($errors)) {
            error_log("Routine validation failed: " . implode(', ', $errors));
            return false;
        }
        
        // Build dynamic update query
        $fields = [];
        $params = ['id' => $id];
        
        $allowedFields = [
            'slug', 'title', 'description', 'difficulty',
            'is_ai_generated', 'is_active'
        ];
        
        foreach ($allowedFields as $field) {
            if (isset($data[$field])) {
                // Special handling for slug
                if ($field === 'slug') {
                    $data[$field] = $this->generateSlug($data[$field], $id);
                }
                $fields[] = "$field = :$field";
                $params[$field] = $data[$field];
            }
        }
        
        if (empty($fields)) {
            return false; // Nothing to update
        }
        
        // Always update timestamp
        $fields[] = "updated_at = NOW()";
        
        try {
            $sql = "UPDATE eye_routines SET " . implode(', ', $fields) . " WHERE id = :id";
            $stmt = $this->pdo->prepare($sql);
            return $stmt->execute($params);
        } catch (Exception $e) {
            error_log("Error updating routine: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Delete routine and all its exercise assignments
     * 
     * @param int $id
     * @return bool
     */
    public function deleteRoutine($id) {
        try {
            $this->pdo->beginTransaction();
            
            // Delete exercise assignments first (foreign key constraint)
            $stmt = $this->pdo->prepare("DELETE FROM eye_routine_exercises WHERE routine_id = :id");
            $stmt->execute(['id' => $id]);
            
            // Delete routine
            $stmt = $this->pdo->prepare("DELETE FROM eye_routines WHERE id = :id");
            $stmt->execute(['id' => $id]);
            
            $this->pdo->commit();
            return true;
        } catch (Exception $e) {
            $this->pdo->rollBack();
            error_log("Error deleting routine: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Toggle active status of routine
     * 
     * @param int $id
     * @return bool
     */
    public function toggleActive($id) {
        try {
            $stmt = $this->pdo->prepare("
                UPDATE eye_routines 
                SET is_active = NOT is_active, updated_at = NOW()
                WHERE id = :id
            ");
            return $stmt->execute(['id' => $id]);
        } catch (Exception $e) {
            error_log("Error toggling routine active status: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Attach exercises to routine (replaces existing assignments)
     * 
     * @param int $routineId
     * @param array $exercisesData Array of ['exercise_id', 'sort_order', 'repetitions', 'override_duration_seconds']
     * @return bool
     */
    public function attachExercisesToRoutine($routineId, $exercisesData) {
        try {
            $this->pdo->beginTransaction();
            
            // Delete existing assignments
            $stmt = $this->pdo->prepare("DELETE FROM eye_routine_exercises WHERE routine_id = :routine_id");
            $stmt->execute(['routine_id' => $routineId]);
            
            // Insert new assignments
            if (!empty($exercisesData)) {
                $stmt = $this->pdo->prepare("
                    INSERT INTO eye_routine_exercises (
                        routine_id, exercise_id, sort_order, 
                        repetitions, override_duration_seconds
                    ) VALUES (
                        :routine_id, :exercise_id, :sort_order,
                        :repetitions, :override_duration_seconds
                    )
                ");
                
                foreach ($exercisesData as $exercise) {
                    // Validate exercise assignment
                    if (empty($exercise['exercise_id']) || empty($exercise['sort_order'])) {
                        throw new Exception("Invalid exercise assignment data");
                    }
                    
                    $stmt->execute([
                        'routine_id' => $routineId,
                        'exercise_id' => $exercise['exercise_id'],
                        'sort_order' => $exercise['sort_order'],
                        'repetitions' => $exercise['repetitions'] ?? 1,
                        'override_duration_seconds' => $exercise['override_duration_seconds'] ?? null
                    ]);
                }
            }
            
            // Recalculate total duration
            $this->recalculateTotalDuration($routineId);
            
            $this->pdo->commit();
            return true;
        } catch (Exception $e) {
            $this->pdo->rollBack();
            error_log("Error attaching exercises to routine: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Recalculate and update total duration for routine
     * 
     * @param int $routineId
     * @return int Calculated total duration in seconds
     */
    public function recalculateTotalDuration($routineId) {
        try {
            // Calculate total using SQL
            $stmt = $this->pdo->prepare("
                SELECT SUM(
                    re.repetitions * 
                    COALESCE(re.override_duration_seconds, e.default_duration_seconds)
                ) as total
                FROM eye_routine_exercises re
                JOIN eye_exercises e ON re.exercise_id = e.id
                WHERE re.routine_id = :routine_id
            ");
            $stmt->execute(['routine_id' => $routineId]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            
            $total = (int)($result['total'] ?? 0);
            
            // Update routine
            $stmt = $this->pdo->prepare("
                UPDATE eye_routines 
                SET total_duration_seconds = :total, updated_at = NOW()
                WHERE id = :id
            ");
            $stmt->execute([
                'total' => $total,
                'id' => $routineId
            ]);
            
            return $total;
        } catch (Exception $e) {
            error_log("Error recalculating total duration: " . $e->getMessage());
            return 0;
        }
    }
    
    /**
     * Validate routine data
     * 
     * @param array $data
     * @param int|null $excludeId
     * @return array Array of error messages
     */
    private function validateRoutineData($data, $excludeId = null) {
        $errors = [];
        
        // Title validation
        if (isset($data['title'])) {
            if (empty(trim($data['title']))) {
                $errors[] = "Title is required";
            } elseif (strlen($data['title']) < 3) {
                $errors[] = "Title must be at least 3 characters";
            } elseif (strlen($data['title']) > 190) {
                $errors[] = "Title must not exceed 190 characters";
            }
        }
        
        // Slug validation (if provided)
        if (isset($data['slug']) && !empty($data['slug'])) {
            if (!preg_match('/^[a-z0-9-]+$/', $data['slug'])) {
                $errors[] = "Slug can only contain lowercase letters, numbers, and hyphens";
            }
            if (strlen($data['slug']) > 190) {
                $errors[] = "Slug must not exceed 190 characters";
            }
        }
        
        // Description validation
        if (isset($data['description']) && strlen($data['description']) > 1000) {
            $errors[] = "Description must not exceed 1000 characters";
        }
        
        // Difficulty validation
        if (isset($data['difficulty'])) {
            $validDifficulties = ['easy', 'medium', 'hard'];
            if (!in_array($data['difficulty'], $validDifficulties)) {
                $errors[] = "Difficulty must be one of: " . implode(', ', $validDifficulties);
            }
        }
        
        return $errors;
    }
    
    /**
     * Generate unique slug from title
     * 
     * @param string $title
     * @param int|null $excludeId
     * @return string
     */
    private function generateSlug($title, $excludeId = null) {
        // Convert to lowercase
        $slug = strtolower($title);
        
        // Replace spaces and special characters with hyphens
        $slug = preg_replace('/[^a-z0-9]+/', '-', $slug);
        
        // Remove consecutive hyphens
        $slug = preg_replace('/-+/', '-', $slug);
        
        // Trim hyphens from start and end
        $slug = trim($slug, '-');
        
        // Ensure uniqueness
        $originalSlug = $slug;
        $counter = 1;
        
        while ($this->slugExists($slug, $excludeId)) {
            $counter++;
            $slug = $originalSlug . '-' . $counter;
        }
        
        return $slug;
    }
    
    /**
     * Check if slug exists in database
     * 
     * @param string $slug
     * @param int|null $excludeId
     * @return bool
     */
    private function slugExists($slug, $excludeId = null) {
        try {
            if ($excludeId) {
                $stmt = $this->pdo->prepare("
                    SELECT COUNT(*) FROM eye_routines 
                    WHERE slug = :slug AND id != :exclude_id
                ");
                $stmt->execute(['slug' => $slug, 'exclude_id' => $excludeId]);
            } else {
                $stmt = $this->pdo->prepare("
                    SELECT COUNT(*) FROM eye_routines 
                    WHERE slug = :slug
                ");
                $stmt->execute(['slug' => $slug]);
            }
            
            return $stmt->fetchColumn() > 0;
        } catch (Exception $e) {
            error_log("Error checking slug existence: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get statistics about routines
     * 
     * @return array
     */
    public function getStatistics() {
        try {
            $stats = [];
            
            // Total count
            $stmt = $this->pdo->query("SELECT COUNT(*) FROM eye_routines");
            $stats['total'] = $stmt->fetchColumn();
            
            // Active count
            $stmt = $this->pdo->query("SELECT COUNT(*) FROM eye_routines WHERE is_active = 1");
            $stats['active'] = $stmt->fetchColumn();
            
            // Inactive count
            $stmt = $this->pdo->query("SELECT COUNT(*) FROM eye_routines WHERE is_active = 0");
            $stats['inactive'] = $stmt->fetchColumn();
            
            // AI generated count
            $stmt = $this->pdo->query("SELECT COUNT(*) FROM eye_routines WHERE is_ai_generated = 1");
            $stats['ai_generated'] = $stmt->fetchColumn();
            
            // Total exercises across all routines
            $stmt = $this->pdo->query("SELECT COUNT(*) FROM eye_routine_exercises");
            $stats['total_exercise_assignments'] = $stmt->fetchColumn();
            
            // By difficulty
            $stmt = $this->pdo->query("
                SELECT difficulty, COUNT(*) as count 
                FROM eye_routines 
                GROUP BY difficulty
            ");
            $stats['by_difficulty'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            return $stats;
        } catch (Exception $e) {
            error_log("Error getting routine statistics: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get available exercises for assignment (not in routine)
     * 
     * @param int|null $routineId Exclude exercises already in this routine
     * @param bool $activeOnly Only return active exercises
     * @return array
     */
    public function getAvailableExercises($routineId = null, $activeOnly = true) {
        try {
            $sql = "SELECT * FROM eye_exercises WHERE 1=1";
            $params = [];
            
            if ($activeOnly) {
                $sql .= " AND is_active = 1";
            }
            
            if ($routineId) {
                $sql .= " AND id NOT IN (
                    SELECT exercise_id 
                    FROM eye_routine_exercises 
                    WHERE routine_id = :routine_id
                )";
                $params['routine_id'] = $routineId;
            }
            
            $sql .= " ORDER BY category ASC, title ASC";
            
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            error_log("Error fetching available exercises: " . $e->getMessage());
            return [];
        }
    }
}
