<?php

/**
 * Magazine Controller
 * Handles magazine CRUD operations, PDF uploads, and category management
 */

class MagazineController {
    private $pdo;
    
    public function __construct($pdo) {
        $this->pdo = $pdo;
    }
    
    /**
     * Get all magazines with optional filters
     * 
     * @param array $filters ['category_id', 'year', 'is_pro_only', 'is_public']
     * @return array
     */
    public function getAllMagazines($filters = []) {
        $sql = "SELECT m.*, 
                COUNT(DISTINCT ccm.category_id) as category_count
                FROM magazines m
                LEFT JOIN content_category_magazine ccm ON m.id = ccm.magazine_id
                WHERE 1=1";
        
        $params = [];
        
        if (isset($filters['category_id'])) {
            $sql .= " AND m.id IN (
                SELECT magazine_id FROM content_category_magazine 
                WHERE category_id = :category_id
            )";
            $params['category_id'] = $filters['category_id'];
        }
        
        if (isset($filters['year'])) {
            $sql .= " AND m.year = :year";
            $params['year'] = $filters['year'];
        }
        
        if (isset($filters['is_pro_only'])) {
            $sql .= " AND m.is_pro_only = :is_pro_only";
            $params['is_pro_only'] = $filters['is_pro_only'];
        }
        
        if (isset($filters['is_public'])) {
            $sql .= " AND m.is_public = :is_public";
            $params['is_public'] = $filters['is_public'];
        }
        
        $sql .= " GROUP BY m.id ORDER BY m.year DESC, m.month DESC, m.issue_number DESC";
        
        try {
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            error_log("Error fetching magazines: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get magazine by ID with categories
     * 
     * @param int $id
     * @return array|null
     */
    public function getMagazineById($id) {
        try {
            $stmt = $this->pdo->prepare("SELECT * FROM magazines WHERE id = :id");
            $stmt->execute(['id' => $id]);
            $magazine = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($magazine) {
                // Get categories
                $magazine['categories'] = $this->getMagazineCategories($id);
            }
            
            return $magazine;
        } catch (Exception $e) {
            error_log("Error fetching magazine: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Get categories for a magazine
     * 
     * @param int $magazineId
     * @return array
     */
    public function getMagazineCategories($magazineId) {
        try {
            $stmt = $this->pdo->prepare("
                SELECT cc.* 
                FROM content_categories cc
                JOIN content_category_magazine ccm ON cc.id = ccm.category_id
                WHERE ccm.magazine_id = :magazine_id
                ORDER BY cc.category_name
            ");
            $stmt->execute(['magazine_id' => $magazineId]);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            error_log("Error fetching magazine categories: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Create new magazine
     * 
     * @param array $data
     * @return int|false Magazine ID or false
     */
    public function createMagazine($data) {
        try {
            $stmt = $this->pdo->prepare("
                INSERT INTO magazines (
                    title, description, issue_number, year, month,
                    pdf_file_path, cover_image_path, file_size,
                    is_public, is_pro_only, publication_date, created_at
                ) VALUES (
                    :title, :description, :issue_number, :year, :month,
                    :pdf_file_path, :cover_image_path, :file_size,
                    :is_public, :is_pro_only, :publication_date, NOW()
                )
            ");
            
            $stmt->execute([
                'title' => $data['title'],
                'description' => $data['description'] ?? null,
                'issue_number' => $data['issue_number'] ?? null,
                'year' => $data['year'],
                'month' => $data['month'] ?? null,
                'pdf_file_path' => $data['pdf_file_path'],
                'cover_image_path' => $data['cover_image_path'] ?? null,
                'file_size' => $data['file_size'] ?? null,
                'is_public' => $data['is_public'] ?? 1,
                'is_pro_only' => $data['is_pro_only'] ?? 0,
                'publication_date' => $data['publication_date'] ?? date('Y-m-d')
            ]);
            
            return $this->pdo->lastInsertId();
        } catch (Exception $e) {
            error_log("Error creating magazine: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Update magazine
     * 
     * @param int $id
     * @param array $data
     * @return bool
     */
    public function updateMagazine($id, $data) {
        try {
            $fields = [];
            $params = ['id' => $id];
            
            $allowed = [
                'title', 'description', 'issue_number', 'year', 'month',
                'pdf_file_path', 'cover_image_path', 'file_size',
                'is_public', 'is_pro_only', 'publication_date'
            ];
            
            foreach ($allowed as $field) {
                if (isset($data[$field])) {
                    $fields[] = "$field = :$field";
                    $params[$field] = $data[$field];
                }
            }
            
            if (empty($fields)) {
                return false;
            }
            
            $sql = "UPDATE magazines SET " . implode(', ', $fields) . ", updated_at = NOW() WHERE id = :id";
            $stmt = $this->pdo->prepare($sql);
            return $stmt->execute($params);
        } catch (Exception $e) {
            error_log("Error updating magazine: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Delete magazine and its files
     * 
     * @param int $id
     * @return bool
     */
    public function deleteMagazine($id) {
        try {
            // Get magazine to delete files
            $magazine = $this->getMagazineById($id);
            if (!$magazine) {
                return false;
            }
            
            // Delete database record
            $stmt = $this->pdo->prepare("DELETE FROM magazines WHERE id = :id");
            $result = $stmt->execute(['id' => $id]);
            
            if ($result) {
                // Delete PDF file
                if (!empty($magazine['pdf_file_path'])) {
                    deleteFile($magazine['pdf_file_path']);
                }
                
                // Delete cover image
                if (!empty($magazine['cover_image_path'])) {
                    deleteFile($magazine['cover_image_path']);
                }
            }
            
            return $result;
        } catch (Exception $e) {
            error_log("Error deleting magazine: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Assign categories to magazine
     * 
     * @param int $magazineId
     * @param array $categoryIds
     * @return bool
     */
    public function assignCategories($magazineId, $categoryIds) {
        try {
            // Delete existing categories
            $stmt = $this->pdo->prepare("DELETE FROM content_category_magazine WHERE magazine_id = :magazine_id");
            $stmt->execute(['magazine_id' => $magazineId]);
            
            // Insert new categories
            if (!empty($categoryIds)) {
                $stmt = $this->pdo->prepare("
                    INSERT INTO content_category_magazine (magazine_id, category_id)
                    VALUES (:magazine_id, :category_id)
                ");
                
                foreach ($categoryIds as $categoryId) {
                    $stmt->execute([
                        'magazine_id' => $magazineId,
                        'category_id' => $categoryId
                    ]);
                }
            }
            
            return true;
        } catch (Exception $e) {
            error_log("Error assigning categories: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Increment download count
     * 
     * @param int $id
     * @return bool
     */
    public function incrementDownloadCount($id) {
        try {
            $stmt = $this->pdo->prepare("
                UPDATE magazines 
                SET download_count = download_count + 1 
                WHERE id = :id
            ");
            return $stmt->execute(['id' => $id]);
        } catch (Exception $e) {
            error_log("Error incrementing download count: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get all categories
     * 
     * @return array
     */
    public function getAllCategories() {
        try {
            $stmt = $this->pdo->query("
                SELECT * FROM content_categories 
                ORDER BY display_order ASC, category_name ASC
            ");
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            error_log("Error fetching categories: " . $e->getMessage());
            return [];
        }
    }
}
