<?php

/**
 * Settings Controller
 * Handles admin settings management for SEO, marketing, email, and site config
 */

class SettingsController {
    private $pdo;
    
    public function __construct($pdo) {
        $this->pdo = $pdo;
    }
    
    /**
     * Get all settings grouped by category
     * 
     * @return array
     */
    public function getAllSettings() {
        return loadSiteSettings();
    }
    
    /**
     * Get settings by category/prefix
     * 
     * @param string $prefix
     * @return array
     */
    public function getSettingsByCategory($prefix) {
        return getSettingsByPrefix($prefix);
    }
    
    /**
     * Update multiple settings
     * 
     * @param array $settings
     * @return bool
     */
    public function updateSettings($settings) {
        try {
            $this->pdo->beginTransaction();
            
            foreach ($settings as $key => $value) {
                $stmt = $this->pdo->prepare("
                    INSERT INTO site_settings (setting_key, setting_value, updated_at)
                    VALUES (:key, :value, NOW())
                    ON DUPLICATE KEY UPDATE 
                        setting_value = :value,
                        updated_at = NOW()
                ");
                $stmt->execute([
                    'key' => $key,
                    'value' => $value
                ]);
            }
            
            $this->pdo->commit();
            return true;
        } catch (Exception $e) {
            $this->pdo->rollBack();
            error_log("Error updating settings: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Initialize default settings if they don't exist
     * 
     * @return bool
     */
    public function initializeDefaults() {
        $defaults = [
            // SEO Settings
            'default_meta_title' => 'Enterlife - Eye Care, Focus & Wellness Platform',
            'default_meta_description' => 'Improve your vision and mental focus with science-backed exercises, tools, and wellness content.',
            'default_og_image' => '/assets/img/og-default.jpg',
            'default_twitter_card_type' => 'summary_large_image',
            'site_schema_org_type' => 'WebSite',
            'robots_txt_content' => "User-agent: *\nAllow: /",
            
            // Marketing & Tracking
            'tracking_enabled' => '1',
            'ga4_measurement_id' => '',
            'gtm_container_id' => '',
            'adsense_client_id' => '',
            'adsense_code_head' => '',
            'adsense_code_body' => '',
            'facebook_pixel_id' => '',
            'custom_head_scripts' => '',
            'custom_body_scripts' => '',
            
            // Email & Newsletter
            'smtp_host' => '',
            'smtp_port' => '587',
            'smtp_username' => '',
            'smtp_password' => '',
            'newsletter_from_name' => 'Enterlife Team',
            'newsletter_from_email' => 'newsletter@enterlife.com',
            'newsletter_enabled' => '1',
            
            // General Site
            'site_name' => 'Enterlife',
            'site_tagline' => 'Vision & Focus Wellness Platform',
            'contact_email' => 'support@enterlife.com',
            'maintenance_mode' => '0'
        ];
        
        try {
            foreach ($defaults as $key => $value) {
                $stmt = $this->pdo->prepare("
                    INSERT IGNORE INTO site_settings (setting_key, setting_value, created_at)
                    VALUES (:key, :value, NOW())
                ");
                $stmt->execute([
                    'key' => $key,
                    'value' => $value
                ]);
            }
            return true;
        } catch (Exception $e) {
            error_log("Error initializing defaults: " . $e->getMessage());
            return false;
        }
    }
}
