<?php

/**
 * Site Settings Management
 * Functions to load, get, and update site-wide settings from site_settings table
 */

/**
 * Load all site settings into an array
 * Returns key-value pairs from site_settings table
 * 
 * @return array
 */
function loadSiteSettings() {
    global $pdo;
    static $cachedSettings = null;
    
    // Return cached settings if already loaded
    if ($cachedSettings !== null) {
        return $cachedSettings;
    }
    
    try {
        $stmt = $pdo->query("SELECT setting_key, setting_value FROM site_settings");
        $settings = [];
        
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            $settings[$row['setting_key']] = $row['setting_value'];
        }
        
        $cachedSettings = $settings;
        return $settings;
    } catch (Exception $e) {
        error_log("Error loading site settings: " . $e->getMessage());
        return [];
    }
}

/**
 * Get a specific setting value
 * 
 * @param string $key Setting key
 * @param mixed $default Default value if setting doesn't exist
 * @return mixed
 */
function getSetting($key, $default = null) {
    global $siteSettings;
    
    if (!isset($siteSettings)) {
        $siteSettings = loadSiteSettings();
    }
    
    return $siteSettings[$key] ?? $default;
}

/**
 * Update or create a setting
 * 
 * @param string $key
 * @param string $value
 * @return bool
 */
function updateSetting($key, $value) {
    global $pdo, $siteSettings;
    
    try {
        $stmt = $pdo->prepare("
            INSERT INTO site_settings (setting_key, setting_value, updated_at)
            VALUES (:key, :value, NOW())
            ON DUPLICATE KEY UPDATE 
                setting_value = :value,
                updated_at = NOW()
        ");
        $result = $stmt->execute([
            'key' => $key,
            'value' => $value
        ]);
        
        // Update cache
        if ($result && isset($siteSettings)) {
            $siteSettings[$key] = $value;
        }
        
        return $result;
    } catch (Exception $e) {
        error_log("Error updating setting: " . $e->getMessage());
        return false;
    }
}

/**
 * Update multiple settings at once
 * 
 * @param array $settings Key-value pairs
 * @return bool
 */
function updateSettings($settings) {
    $success = true;
    
    foreach ($settings as $key => $value) {
        if (!updateSetting($key, $value)) {
            $success = false;
        }
    }
    
    return $success;
}

/**
 * Delete a setting
 * 
 * @param string $key
 * @return bool
 */
function deleteSetting($key) {
    global $pdo, $siteSettings;
    
    try {
        $stmt = $pdo->prepare("DELETE FROM site_settings WHERE setting_key = :key");
        $result = $stmt->execute(['key' => $key]);
        
        // Update cache
        if ($result && isset($siteSettings)) {
            unset($siteSettings[$key]);
        }
        
        return $result;
    } catch (Exception $e) {
        error_log("Error deleting setting: " . $e->getMessage());
        return false;
    }
}

/**
 * Get settings by prefix (e.g., all 'ga4_*' settings)
 * 
 * @param string $prefix
 * @return array
 */
function getSettingsByPrefix($prefix) {
    global $siteSettings;
    
    if (!isset($siteSettings)) {
        $siteSettings = loadSiteSettings();
    }
    
    $filtered = [];
    foreach ($siteSettings as $key => $value) {
        if (strpos($key, $prefix) === 0) {
            $filtered[$key] = $value;
        }
    }
    
    return $filtered;
}

/**
 * Render tracking/analytics scripts in HTML head
 * 
 * @return void
 */
function renderHeadScripts() {
    global $siteSettings;
    
    if (!isset($siteSettings)) {
        $siteSettings = loadSiteSettings();
    }
    
    // Only render if tracking is enabled
    if (empty($siteSettings['tracking_enabled']) || $siteSettings['tracking_enabled'] !== '1') {
        return;
    }
    
    // Google Tag Manager
    if (!empty($siteSettings['gtm_container_id'])) {
        echo "<!-- Google Tag Manager -->\n";
        echo "<script>(function(w,d,s,l,i){w[l]=w[l]||[];w[l].push({'gtm.start':\n";
        echo "new Date().getTime(),event:'gtm.js'});var f=d.getElementsByTagName(s)[0],\n";
        echo "j=d.createElement(s),dl=l!='dataLayer'?'&l='+l:'';j.async=true;j.src=\n";
        echo "'https://www.googletagmanager.com/gtm.js?id='+i+dl;f.parentNode.insertBefore(j,f);\n";
        echo "})(window,document,'script','dataLayer','" . e($siteSettings['gtm_container_id']) . "');</script>\n";
        echo "<!-- End Google Tag Manager -->\n";
    }
    
    // Google Analytics 4
    if (!empty($siteSettings['ga4_measurement_id'])) {
        echo "<!-- Google Analytics -->\n";
        echo "<script async src=\"https://www.googletagmanager.com/gtag/js?id=" . e($siteSettings['ga4_measurement_id']) . "\"></script>\n";
        echo "<script>\n";
        echo "  window.dataLayer = window.dataLayer || [];\n";
        echo "  function gtag(){dataLayer.push(arguments);}\n";
        echo "  gtag('js', new Date());\n";
        echo "  gtag('config', '" . e($siteSettings['ga4_measurement_id']) . "');\n";
        echo "</script>\n";
        echo "<!-- End Google Analytics -->\n";
    }
    
    // Google Adsense
    if (!empty($siteSettings['adsense_code_head'])) {
        echo $siteSettings['adsense_code_head'] . "\n";
    }
    
    // Facebook Pixel
    if (!empty($siteSettings['facebook_pixel_id'])) {
        echo "<!-- Facebook Pixel Code -->\n";
        echo "<script>\n";
        echo "!function(f,b,e,v,n,t,s)\n";
        echo "{if(f.fbq)return;n=f.fbq=function(){n.callMethod?\n";
        echo "n.callMethod.apply(n,arguments):n.queue.push(arguments)};\n";
        echo "if(!f._fbq)f._fbq=n;n.push=n;n.loaded=!0;n.version='2.0';\n";
        echo "n.queue=[];t=b.createElement(e);t.async=!0;\n";
        echo "t.src=v;s=b.getElementsByTagName(e)[0];\n";
        echo "s.parentNode.insertBefore(t,s)}(window, document,'script',\n";
        echo "'https://connect.facebook.net/en_US/fbevents.js');\n";
        echo "fbq('init', '" . e($siteSettings['facebook_pixel_id']) . "');\n";
        echo "fbq('track', 'PageView');\n";
        echo "</script>\n";
        echo "<!-- End Facebook Pixel Code -->\n";
    }
    
    // Custom head scripts
    if (!empty($siteSettings['custom_head_scripts'])) {
        echo $siteSettings['custom_head_scripts'] . "\n";
    }
}

/**
 * Render tracking/analytics scripts in HTML body (GTM noscript, etc.)
 * 
 * @return void
 */
function renderBodyScripts() {
    global $siteSettings;
    
    if (!isset($siteSettings)) {
        $siteSettings = loadSiteSettings();
    }
    
    // Only render if tracking is enabled
    if (empty($siteSettings['tracking_enabled']) || $siteSettings['tracking_enabled'] !== '1') {
        return;
    }
    
    // Google Tag Manager (noscript)
    if (!empty($siteSettings['gtm_container_id'])) {
        echo "<!-- Google Tag Manager (noscript) -->\n";
        echo "<noscript><iframe src=\"https://www.googletagmanager.com/ns.html?id=" . e($siteSettings['gtm_container_id']) . "\"\n";
        echo "height=\"0\" width=\"0\" style=\"display:none;visibility:hidden\"></iframe></noscript>\n";
        echo "<!-- End Google Tag Manager (noscript) -->\n";
    }
    
    // Adsense body scripts
    if (!empty($siteSettings['adsense_code_body'])) {
        echo $siteSettings['adsense_code_body'] . "\n";
    }
    
    // Custom body scripts
    if (!empty($siteSettings['custom_body_scripts'])) {
        echo $siteSettings['custom_body_scripts'] . "\n";
    }
}
