<?php

/**
 * File Upload Helper Functions
 * Handles file uploads for magazines, images, PDFs, etc.
 */

/**
 * Upload a file to a specified directory
 * 
 * @param array $file $_FILES array element
 * @param string $targetDir Target directory (e.g., 'uploads/magazines')
 * @param array $allowedTypes Allowed MIME types
 * @param int $maxSize Max file size in bytes (default 10MB)
 * @return array ['success' => bool, 'filename' => string|null, 'error' => string|null]
 */
function uploadFile($file, $targetDir, $allowedTypes = [], $maxSize = 10485760) {
    // Check if file was uploaded
    if (!isset($file['tmp_name']) || empty($file['tmp_name'])) {
        return ['success' => false, 'filename' => null, 'error' => 'No file uploaded'];
    }
    
    // Check for upload errors
    if ($file['error'] !== UPLOAD_ERR_OK) {
        return ['success' => false, 'filename' => null, 'error' => 'Upload error: ' . $file['error']];
    }
    
    // Check file size
    if ($file['size'] > $maxSize) {
        $maxSizeMB = round($maxSize / 1048576, 2);
        return ['success' => false, 'filename' => null, 'error' => "File too large. Max size: {$maxSizeMB}MB"];
    }
    
    // Check file type
    if (!empty($allowedTypes)) {
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mimeType = finfo_file($finfo, $file['tmp_name']);
        finfo_close($finfo);
        
        if (!in_array($mimeType, $allowedTypes)) {
            return ['success' => false, 'filename' => null, 'error' => 'Invalid file type'];
        }
    }
    
    // Create target directory if it doesn't exist
    $uploadPath = __DIR__ . '/../../' . trim($targetDir, '/');
    if (!is_dir($uploadPath)) {
        if (!mkdir($uploadPath, 0755, true)) {
            return ['success' => false, 'filename' => null, 'error' => 'Failed to create upload directory'];
        }
    }
    
    // Generate unique filename
    $extension = pathinfo($file['name'], PATHINFO_EXTENSION);
    $filename = uniqid() . '_' . time() . '.' . $extension;
    $targetPath = $uploadPath . '/' . $filename;
    
    // Move uploaded file
    if (move_uploaded_file($file['tmp_name'], $targetPath)) {
        return ['success' => true, 'filename' => $filename, 'error' => null];
    } else {
        return ['success' => false, 'filename' => null, 'error' => 'Failed to move uploaded file'];
    }
}

/**
 * Upload PDF file
 * 
 * @param array $file $_FILES array element
 * @param string $targetDir Target directory
 * @return array
 */
function uploadPDF($file, $targetDir = 'uploads/magazines') {
    return uploadFile($file, $targetDir, ['application/pdf'], 52428800); // 50MB max for PDFs
}

/**
 * Upload image file
 * 
 * @param array $file $_FILES array element
 * @param string $targetDir Target directory
 * @return array
 */
function uploadImage($file, $targetDir = 'uploads/images') {
    $allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
    return uploadFile($file, $targetDir, $allowedTypes, 5242880); // 5MB max for images
}

/**
 * Delete a file
 * 
 * @param string $filePath Relative path from project root
 * @return bool
 */
function deleteFile($filePath) {
    $fullPath = __DIR__ . '/../../' . ltrim($filePath, '/');
    
    if (file_exists($fullPath)) {
        return unlink($fullPath);
    }
    
    return false;
}

/**
 * Get file size in human-readable format
 * 
 * @param int $bytes
 * @return string
 */
function formatFileSize($bytes) {
    $units = ['B', 'KB', 'MB', 'GB'];
    $bytes = max($bytes, 0);
    $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
    $pow = min($pow, count($units) - 1);
    $bytes /= (1 << (10 * $pow));
    
    return round($bytes, 2) . ' ' . $units[$pow];
}

/**
 * Serve file download with proper headers
 * 
 * @param string $filePath Absolute path to file
 * @param string $downloadName Filename to send to browser
 * @param bool $forceDownload Force download dialog (vs inline display)
 * @return void
 */
function serveFile($filePath, $downloadName = null, $forceDownload = true) {
    if (!file_exists($filePath)) {
        http_response_code(404);
        die('File not found');
    }
    
    $downloadName = $downloadName ?? basename($filePath);
    $mimeType = mime_content_type($filePath);
    $fileSize = filesize($filePath);
    
    // Clear output buffer
    if (ob_get_level()) {
        ob_end_clean();
    }
    
    // Set headers
    header('Content-Type: ' . $mimeType);
    header('Content-Length: ' . $fileSize);
    
    if ($forceDownload) {
        header('Content-Disposition: attachment; filename="' . $downloadName . '"');
    } else {
        header('Content-Disposition: inline; filename="' . $downloadName . '"');
    }
    
    header('Cache-Control: no-cache, must-revalidate');
    header('Expires: 0');
    
    // Output file
    readfile($filePath);
    exit;
}

/**
 * Validate and sanitize filename
 * 
 * @param string $filename
 * @return string
 */
function sanitizeFilename($filename) {
    // Remove any path info
    $filename = basename($filename);
    
    // Remove special characters
    $filename = preg_replace('/[^a-zA-Z0-9._-]/', '_', $filename);
    
    return $filename;
}

/**
 * Create thumbnail from image
 * 
 * @param string $sourcePath
 * @param string $targetPath
 * @param int $maxWidth
 * @param int $maxHeight
 * @return bool
 */
function createThumbnail($sourcePath, $targetPath, $maxWidth = 300, $maxHeight = 300) {
    if (!file_exists($sourcePath)) {
        return false;
    }
    
    $imageInfo = getimagesize($sourcePath);
    if (!$imageInfo) {
        return false;
    }
    
    list($width, $height, $type) = $imageInfo;
    
    // Calculate new dimensions
    $ratio = min($maxWidth / $width, $maxHeight / $height);
    $newWidth = (int)($width * $ratio);
    $newHeight = (int)($height * $ratio);
    
    // Create image resource
    switch ($type) {
        case IMAGETYPE_JPEG:
            $source = imagecreatefromjpeg($sourcePath);
            break;
        case IMAGETYPE_PNG:
            $source = imagecreatefrompng($sourcePath);
            break;
        case IMAGETYPE_GIF:
            $source = imagecreatefromgif($sourcePath);
            break;
        default:
            return false;
    }
    
    // Create thumbnail
    $thumbnail = imagecreatetruecolor($newWidth, $newHeight);
    
    // Preserve transparency for PNG
    if ($type == IMAGETYPE_PNG) {
        imagealphablending($thumbnail, false);
        imagesavealpha($thumbnail, true);
    }
    
    imagecopyresampled($thumbnail, $source, 0, 0, 0, 0, $newWidth, $newHeight, $width, $height);
    
    // Save thumbnail
    $result = false;
    switch ($type) {
        case IMAGETYPE_JPEG:
            $result = imagejpeg($thumbnail, $targetPath, 85);
            break;
        case IMAGETYPE_PNG:
            $result = imagepng($thumbnail, $targetPath, 8);
            break;
        case IMAGETYPE_GIF:
            $result = imagegif($thumbnail, $targetPath);
            break;
    }
    
    imagedestroy($source);
    imagedestroy($thumbnail);
    
    return $result;
}
