<?php
require_once __DIR__ . '/../../includes/bootstrap.php';

requireAdmin();

$pdo = getPDO();

// Handle category actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    verify_csrf();
    
    $action = $_POST['action'] ?? '';
    
    if ($action === 'create' || $action === 'update') {
        $id = $_POST['id'] ?? null;
        $categoryName = $_POST['category_name'] ?? '';
        $slug = $_POST['slug'] ?? '';
        $description = $_POST['description'] ?? '';
        $displayOrder = $_POST['display_order'] ?? 0;
        
        if($action === 'create') {
            $stmt = $pdo->prepare("
                INSERT INTO content_categories (category_name, slug, description, display_order, created_at)
                VALUES (:name, :slug, :description, :order, NOW())
            ");
            $stmt->execute([
                'name' => $categoryName,
                'slug' => $slug,
                'description' => $description,
                'order' => $displayOrder
            ]);
            flash('flash_message', 'Category created successfully!', 'success');
        } else {
            $stmt = $pdo->prepare("
                UPDATE content_categories 
                SET category_name = :name, slug = :slug, description = :description, 
                    display_order = :order, updated_at = NOW()
                WHERE id = :id
            ");
            $stmt->execute([
                'name' => $categoryName,
                'slug' => $slug,
                'description' => $description,
                'order' => $displayOrder,
                'id' => $id
            ]);
            flash('flash_message', 'Category updated successfully!', 'success');
        }
    } elseif ($action === 'delete') {
        $id = $_POST['id'] ?? null;
        if ($id) {
            $stmt = $pdo->prepare("DELETE FROM content_categories WHERE id = :id");
            $stmt->execute(['id' => $id]);
            flash('flash_message', 'Category deleted successfully!', 'success');
        }
    }
    
    redirect(base_url('admin/articles-categories.php'));
}

// Get all categories with article/magazine counts
$stmt = $pdo->query("
    SELECT c.*,
           (SELECT COUNT(*) FROM content_category_article WHERE category_id = c.id) as article_count,
           (SELECT COUNT(*) FROM content_category_magazine WHERE category_id = c.id) as magazine_count
    FROM content_categories c
    ORDER BY c.display_order ASC, c.category_name ASC
");
$categories = $stmt->fetchAll(PDO::FETCH_ASSOC);

$pageTitle = 'Categories - Admin';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= e($pageTitle) ?></title>
    <link rel="stylesheet" href="<?= base_url('assets/css/style.css') ?>">
    <style>
        .admin-container { max-width: 1200px; margin: 40px auto; padding: 20px; }
        .admin-nav { background: #f8f9fa; padding: 15px; border-radius: 8px; margin-bottom: 30px; }
        .admin-nav a { margin-right: 15px; padding: 8px 15px; background: #007bff; color: white; text-decoration: none; border-radius: 5px; display: inline-block; margin-bottom: 5px; }
        .admin-nav a:hover { background: #0056b3; }
        .admin-nav a.active { background: #28a745; }
        .btn { padding: 10px 20px; background: #007bff; color: white; border: none; border-radius: 5px; cursor: pointer; text-decoration: none; display: inline-block; }
        .btn:hover { background: #0056b3; }
        .btn-danger { background: #dc3545; }
        .btn-danger:hover { background: #c82333; }
        .btn-sm { padding: 5px 10px; font-size: 12px; }
        table { width: 100%; border-collapse: collapse; margin-top: 20px; }
        th, td { padding: 12px; text-align: left; border-bottom: 1px solid #ddd; }
        th { background: #f8f9fa; font-weight: bold; }
        .alert { padding: 15px; border-radius: 5px; margin-bottom: 20px; }
        .alert-success { background: #d4edda; color: #155724; }
        .alert-danger { background: #f8d7da; color: #721c24; }
        .form-group { margin-bottom: 15px; }
        .form-group label { display: block; font-weight: bold; margin-bottom: 5px; }
        .form-group input, .form-group textarea { width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 5px; }
        .modal { display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 1000; }
        .modal.active { display: flex; align-items: center; justify-content: center; }
        .modal-content { background: white; padding: 30px; border-radius: 10px; max-width: 500px; width: 90%; }
        .modal-header { margin-bottom: 20px; }
        .modal-header h2 { margin: 0; }
        .close { float: right; font-size: 28px; cursor: pointer; color: #999; }
        .close:hover { color: #333; }
    </style>
</head>
<body>
    <div class="admin-container">
        <h1>Admin Panel - Content Categories</h1>
        
        <?php include 'partials/admin-nav.php'; ?>
        
        <?php flash('flash_message'); ?>
        
        <div style="margin-bottom: 20px;">
            <button class="btn" onclick="openModal('create')">+ Create New Category</button>
        </div>
        
        <table>
            <thead>
                <tr>
                    <th>ID</th>
                    <th>Name</th>
                    <th>Slug</th>
                    <th>Articles</th>
                    <th>Magazines</th>
                    <th>Display Order</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($categories as $cat): ?>
                <tr>
                    <td><?= e($cat['id']) ?></td>
                    <td><?= e($cat['category_name']) ?></td>
                    <td><?= e($cat['slug']) ?></td>
                    <td><?= e($cat['article_count']) ?></td>
                    <td><?= e($cat['magazine_count']) ?></td>
                    <td><?= e($cat['display_order']) ?></td>
                    <td>
                        <button class="btn btn-sm" onclick='editCategory(<?= json_encode($cat) ?>)'>Edit</button>
                        <form method="POST" style="display:inline;" onsubmit="return confirm('Delete this category?');">
                            <?= csrf_field() ?>
                            <input type="hidden" name="action" value="delete">
                            <input type="hidden" name="id" value="<?= e($cat['id']) ?>">
                            <button type="submit" class="btn btn-sm btn-danger">Delete</button>
                        </form>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
    
    <!-- Modal -->
    <div id="categoryModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <span class="close" onclick="closeModal()">&times;</span>
                <h2 id="modalTitle">Create Category</h2>
            </div>
            <form method="POST">
                <?= csrf_field() ?>
                <input type="hidden" name="action" id="formAction" value="create">
                <input type="hidden" name="id" id="categoryId">
                
                <div class="form-group">
                    <label for="category_name">Category Name</label>
                    <input type="text" id="category_name" name="category_name" required>
                </div>
                
                <div class="form-group">
                    <label for="slug">Slug</label>
                    <input type="text" id="slug" name="slug" required>
                </div>
                
                <div class="form-group">
                    <label for="description">Description</label>
                    <textarea id="description" name="description" rows="3"></textarea>
                </div>
                
                <div class="form-group">
                    <label for="display_order">Display Order</label>
                    <input type="number" id="display_order" name="display_order" value="0">
                </div>
                
                <button type="submit" class="btn">Save Category</button>
            </form>
        </div>
    </div>
    
    <script>
        function openModal(action) {
            document.getElementById('categoryModal').classList.add('active');
            document.getElementById('formAction').value = action;
            document.getElementById('modalTitle').textContent = action === 'create' ? 'Create Category' : 'Edit Category';
            
            if (action === 'create') {
                document.getElementById('categoryId').value = '';
                document.getElementById('category_name').value = '';
                document.getElementById('slug').value = '';
                document.getElementById('description').value = '';
                document.getElementById('display_order').value = '0';
            }
        }
        
        function editCategory(cat) {
            openModal('update');
            document.getElementById('categoryId').value = cat.id;
            document.getElementById('category_name').value = cat.category_name;
            document.getElementById('slug').value = cat.slug;
            document.getElementById('description').value = cat.description || '';
            document.getElementById('display_order').value = cat.display_order;
        }
        
        function closeModal() {
            document.getElementById('categoryModal').classList.remove('active');
        }
        
        // Auto-generate slug from name
        document.getElementById('category_name').addEventListener('input', function(e) {
            if (document.getElementById('formAction').value === 'create') {
                document.getElementById('slug').value = e.target.value.toLowerCase().replace(/[^a-z0-9]+/g, '-').replace(/(^-|-$)/g, '');
            }
        });
    </script>
</body>
</html>
