<?php
require_once __DIR__ . '/../../includes/bootstrap.php';

requireAdmin();

$routineController = new EyeRoutineController(getPDO());

// Handle quick actions
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    verify_csrf();
    
    $action = $_POST['action'];
    $id = $_POST['id'] ?? null;
    
    if ($action === 'toggle_active' && $id) {
        if ($routineController->toggleActive($id)) {
            flash('flash_message', 'Routine status updated!', 'success');
        } else {
            flash('flash_message', 'Error updating status.', 'danger');
        }
    } elseif ($action === 'delete' && $id) {
        if ($routineController->deleteRoutine($id)) {
            flash('flash_message', 'Routine deleted successfully!', 'success');
        } else {
            flash('flash_message', 'Error deleting routine.', 'danger');
        }
    }
    
    redirect(base_url('admin/eye-routines.php'));
}

// Get filters
$filters = [];
if (!empty($_GET['difficulty'])) {
    $filters['difficulty'] = $_GET['difficulty'];
}
if (isset($_GET['is_active']) && $_GET['is_active'] !== '') {
    $filters['is_active'] = $_GET['is_active'];
}
if (isset($_GET['is_ai_generated']) && $_GET['is_ai_generated'] !== '') {
    $filters['is_ai_generated'] = $_GET['is_ai_generated'];
}
if (!empty($_GET['search'])) {
    $filters['search'] = $_GET['search'];
}

$routines = $routineController->getAllRoutines($filters);
$stats = $routineController->getStatistics();

$pageTitle = 'Eye Routines - Admin';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= e($pageTitle) ?></title>
    <link rel="stylesheet" href="<?= base_url('assets/css/style.css') ?>">
    <style>
        .admin-container { max-width: 1400px; margin: 40px auto; padding: 20px; }
        .admin-nav { background: #f8f9fa; padding: 15px; border-radius: 8px; margin-bottom: 30px; }
        .admin-nav a { margin-right: 15px; padding: 8px 15px; background: #007bff; color: white; text-decoration: none; border-radius: 5px; display: inline-block; margin-bottom: 5px; }
        .admin-nav a:hover { background: #0056b3; }
        .admin-nav a.active { background: #28a745; }
        .stats-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 15px; margin-bottom: 30px; }
        .stat-card { background: #f8f9fa; border: 1px solid #ddd; border-radius: 8px; padding: 15px; text-align: center; }
        .stat-card h3 { margin: 0; font-size: 28px; color: #007bff; }
        .stat-card p { margin: 5px 0 0; color: #666; font-size: 14px; }
        .filters { background: #f8f9fa; padding: 15px; border-radius: 8px; margin-bottom: 20px; display: flex; gap: 10px; flex-wrap: wrap; align-items: center; }
        .filters select, .filters input { padding: 8px; border: 1px solid #ddd; border-radius: 5px; }
        .btn { padding: 10px 20px; background: #007bff; color: white; border: none; border-radius: 5px; cursor: pointer; text-decoration: none; display: inline-block; }
        .btn:hover { background: #0056b3; }
        .btn-danger { background: #dc3545; }
        .btn-danger:hover { background: #c82333; }
        .btn-sm { padding: 5px 10px; font-size: 12px; }
        table { width: 100%; border-collapse: collapse; margin-top: 20px; }
        th, td { padding: 12px; text-align: left; border-bottom: 1px solid #ddd; }
        th { background: #f8f9fa; font-weight: bold; }
        tr:hover { background: #f8f9fa; }
        .badge { padding: 4px 8px; border-radius: 3px; font-size: 11px; font-weight: bold; text-transform: uppercase; }
        .badge-easy { background: #28a745; color: white; }
        .badge-medium { background: #ffc107; color: #333; }
        .badge-hard { background: #dc3545; color: white; }
        .badge-ai { background: #6f42c1; color: white; }
        .badge-active { background: #28a745; color: white; }
        .badge-inactive { background: #dc3545; color: white; }
        .alert { padding: 15px; border-radius: 5px; margin-bottom: 20px; }
        .alert-success { background: #d4edda; color: #155724; }
        .alert-danger { background: #f8d7da; color: #721c24; }
        .toggle-switch { position: relative; display: inline-block; width: 50px; height: 24px; }
        .toggle-switch input { opacity: 0; width: 0; height: 0; }
        .slider { position: absolute; cursor: pointer; top: 0; left: 0; right: 0; bottom: 0; background-color: #ccc; transition: .4s; border-radius: 24px; }
        .slider:before { position: absolute; content: ""; height: 16px; width: 16px; left: 4px; bottom: 4px; background-color: white; transition: .4s; border-radius: 50%; }
        input:checked + .slider { background-color: #28a745; }
        input:checked + .slider:before { transform: translateX(26px); }
        .modal { display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.6); z-index: 1000; align-items: center; justify-content: center; }
        .modal.active { display: flex; }
        .modal-content { background: white; padding: 30px; border-radius: 10px; max-width: 600px; width: 90%; max-height: 80vh; overflow-y: auto; }
        .modal-content h2 { margin-top: 0; }
        .exercise-list { list-style: none; padding: 0; margin: 20px 0; }
        .exercise-list li { padding: 10px; border: 1px solid #ddd; border-radius: 5px; margin-bottom: 10px; }
        .exercise-list li strong { display: block; color: #007bff; }
    </style>
</head>
<body>
    <div class="admin-container">
        <h1>Admin Panel - Eye Routines</h1>
        
        <?php include 'partials/admin-nav.php'; ?>
        
        <?php flash('flash_message'); ?>
        
        <div class="stats-grid">
            <div class="stat-card">
                <h3><?= e($stats['total']) ?></h3>
                <p>Total Routines</p>
            </div>
            <div class="stat-card">
                <h3><?= e($stats['active']) ?></h3>
                <p>Active</p>
            </div>
            <div class="stat-card">
                <h3><?= e($stats['inactive']) ?></h3>
                <p>Inactive</p>
            </div>
            <div class="stat-card">
                <h3><?= e($stats['ai_generated']) ?></h3>
                <p>AI Generated</p>
            </div>
        </div>
        
        <div style="margin-bottom: 20px;">
            <a href="<?= base_url('admin/eye-routines-edit.php') ?>" class="btn">+ Create New Routine</a>
        </div>
        
        <div class="filters">
            <form method="GET" style="display: flex; gap: 10px; flex-wrap: wrap; width: 100%;">
                <select name="difficulty" onchange="this.form.submit()">
                    <option value="">All Difficulties</option>
                    <option value="easy" <?= ($_GET['difficulty'] ?? '') === 'easy' ? 'selected' : '' ?>>Easy</option>
                    <option value="medium" <?= ($_GET['difficulty'] ?? '') === 'medium' ? 'selected' : '' ?>>Medium</option>
                    <option value="hard" <?= ($_GET['difficulty'] ?? '') === 'hard' ? 'selected' : '' ?>>Hard</option>
                </select>
                
                <select name="is_active" onchange="this.form.submit()">
                    <option value="">All Status</option>
                    <option value="1" <?= ($_GET['is_active'] ?? '') === '1' ? 'selected' : '' ?>>Active Only</option>
                    <option value="0" <?= ($_GET['is_active'] ?? '') === '0' ? 'selected' : '' ?>>Inactive Only</option>
                </select>
                
                <select name="is_ai_generated" onchange="this.form.submit()">
                    <option value="">All Types</option>
                    <option value="0" <?= ($_GET['is_ai_generated'] ?? '') === '0' ? 'selected' : '' ?>>Manual Only</option>
                    <option value="1" <?= ($_GET['is_ai_generated'] ?? '') === '1' ? 'selected' : '' ?>>AI Generated Only</option>
                </select>
                
                <input type="text" name="search" placeholder="Search by title..." 
                       value="<?= e($_GET['search'] ?? '') ?>" style="flex: 1; min-width: 200px;">
                
                <button type="submit" class="btn">Filter</button>
                <?php if (!empty($_GET)): ?>
                    <a href="<?= base_url('admin/eye-routines.php') ?>" class="btn" style="background: #6c757d;">Clear</a>
                <?php endif; ?>
            </form>
        </div>
        
        <table>
            <thead>
                <tr>
                    <th>ID</th>
                    <th>Title</th>
                    <th>Slug</th>
                    <th>Exercises</th>
                    <th>Difficulty</th>
                    <th>Duration</th>
                    <th>Type</th>
                    <th>Active</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($routines)): ?>
                    <tr>
                        <td colspan="9" style="text-align: center; padding: 40px;">
                            No routines found. <a href="<?= base_url('admin/eye-routines-edit.php') ?>">Create your first routine</a>
                        </td>
                    </tr>
                <?php else: ?>
                    <?php foreach ($routines as $routine): ?>
                    <tr>
                        <td><?= e($routine['id']) ?></td>
                        <td>
                            <strong><?= e($routine['title']) ?></strong>
                            <?php if (!empty($routine['description'])): ?>
                                <br><small style="color: #666;"><?= e(substr($routine['description'], 0, 60)) ?><?= strlen($routine['description']) > 60 ? '...' : '' ?></small>
                            <?php endif; ?>
                        </td>
                        <td><code><?= e($routine['slug']) ?></code></td>
                        <td><?= e($routine['exercise_count']) ?> exercises</td>
                        <td><span class="badge badge-<?= e($routine['difficulty']) ?>"><?= e($routine['difficulty']) ?></span></td>
                        <td>
                            <?php
                            $total = $routine['total_duration_seconds'];
                            $mins = floor($total / 60);
                            $secs = $total % 60;
                            echo $mins > 0 ? "{$mins}m " : '';
                            echo "{$secs}s";
                            ?>
                        </td>
                        <td>
                            <?php if ($routine['is_ai_generated']): ?>
                                <span class="badge badge-ai">AI</span>
                            <?php else: ?>
                                Manual
                            <?php endif; ?>
                        </td>
                        <td>
                            <form method="POST" style="display: inline;">
                                <?= csrf_field() ?>
                                <input type="hidden" name="action" value="toggle_active">
                                <input type="hidden" name="id" value="<?= e($routine['id']) ?>">
                                <label class="toggle-switch">
                                    <input type="checkbox" <?= $routine['is_active'] ? 'checked' : '' ?> onchange="this.form.submit()">
                                    <span class="slider"></span>
                                </label>
                            </form>
                        </td>
                        <td>
                            <a href="<?= base_url('admin/eye-routines-edit.php?id=' . $routine['id']) ?>" class="btn btn-sm">Edit</a>
                            <button class="btn btn-sm" style="background: #17a2b8;" onclick="previewRoutine(<?= e($routine['id']) ?>)">Preview</button>
                            <form method="POST" style="display: inline;" onsubmit="return confirm('Delete this routine? This cannot be undone.');">
                                <?= csrf_field() ?>
                                <input type="hidden" name="action" value="delete">
                                <input type="hidden" name="id" value="<?= e($routine['id']) ?>">
                                <button type="submit" class="btn btn-sm btn-danger">Delete</button>
                            </form>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
    
    <!-- Preview Modal -->
    <div id="previewModal" class="modal">
        <div class="modal-content">
            <h2 id="modalTitle">Routine Preview</h2>
            <div id="modalBody">
                Loading...
            </div>
            <button class="btn" onclick="closePreview()">Close</button>
        </div>
    </div>
    
    <script>
        function previewRoutine(routineId) {
            const modal = document.getElementById('previewModal');
            modal.classList.add('active');
            
            // Fetch routine details via AJAX
            fetch('<?= base_url('admin/api/get-routine-preview.php') ?>?id=' + routineId)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        document.getElementById('modalTitle').textContent = data.routine.title;
                        
                        let html = '<p><strong>Description:</strong> ' + (data.routine.description || 'No description') + '</p>';
                        html += '<p><strong>Difficulty:</strong> ' + data.routine.difficulty + '</p>';
                        html += '<p><strong>Total Duration:</strong> ' + formatDuration(data.routine.total_duration_seconds) + '</p>';
                        html += '<h3>Exercises (' + data.exercises.length + '):</h3>';
                        html += '<ul class="exercise-list">';
                        
                        data.exercises.forEach((ex, index) => {
                            html += '<li>';
                            html += '<strong>' + (index + 1) + '. ' + ex.exercise_title + '</strong>';
                            html += '<div>Category: ' + ex.exercise_category + ' | Difficulty: ' + ex.exercise_difficulty + '</div>';
                            html += '<div>Repetitions: ' + ex.repetitions + ' × ' + formatDuration(ex.effective_duration) + ' = ' + formatDuration(ex.total_contribution) + '</div>';
                            html += '</li>';
                        });
                        
                        html += '</ul>';
                        document.getElementById('modalBody').innerHTML = html;
                    } else {
                        document.getElementById('modalBody').innerHTML = '<p style="color: red;">Error loading routine preview.</p>';
                    }
                })
                .catch(error => {
                    document.getElementById('modalBody').innerHTML = '<p style="color: red;">Error: ' + error.message + '</p>';
                });
        }
        
        function closePreview() {
            document.getElementById('previewModal').classList.remove('active');
        }
        
        function formatDuration(seconds) {
            const mins = Math.floor(seconds / 60);
            const secs = seconds % 60;
            return (mins > 0 ? mins + 'm ' : '') + secs + 's';
        }
        
        // Close modal on outside click
        document.getElementById('previewModal').addEventListener('click', function(e) {
            if (e.target === this) {
                closePreview();
            }
        });
    </script>
</body>
</html>
