<?php
require_once __DIR__ . '/../../includes/bootstrap.php';

requireAdmin();

$magazineController = new MagazineController(getPDO());
$categories = $magazineController->getAllCategories();

$editMode = false;
$magazine = null;

// Load magazine for editing
if (isset($_GET['id'])) {
    $editMode = true;
    $magazine = $magazineController->getMagazineById($_GET['id']);
    if (!$magazine) {
        flash('flash_message', 'Magazine not found.', 'danger');
        redirect(base_url('admin/magazines.php'));
    }
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    verify_csrf();
    
    $data = [
        'title' => $_POST['title'] ?? '',
        'description' => $_POST['description'] ?? '',
        'issue_number' => $_POST['issue_number'] ?? null,
        'year' => $_POST['year'] ?? date('Y'),
        'month' => $_POST['month'] ?? null,
        'is_public' => isset($_POST['is_public']) ? 1 : 0,
        'is_pro_only' => isset($_POST['is_pro_only']) ? 1 : 0,
        'publication_date' => $_POST['publication_date'] ?? date('Y-m-d')
    ];
    
    $magazineId = $_POST['magazine_id'] ?? null;
    
    // Handle PDF upload
    if (isset($_FILES['pdf_file']) && $_FILES['pdf_file']['error'] === UPLOAD_ERR_OK) {
        $pdfUpload = uploadPDF($_FILES['pdf_file']);
        if ($pdfUpload['success']) {
            $data['pdf_file_path'] = 'uploads/magazines/' . $pdfUpload['filename'];
            $data['file_size'] = $_FILES['pdf_file']['size'];
            
            // Delete old PDF if editing
            if ($editMode && !empty($magazine['pdf_file_path'])) {
                deleteFile($magazine['pdf_file_path']);
            }
        } else {
            flash('flash_message', 'Error uploading PDF: ' . $pdfUpload['error'], 'danger');
            redirect(base_url('admin/magazines-edit.php' . ($magazineId ? '?id=' . $magazineId : '')));
        }
    } elseif ($editMode) {
        // Keep existing file path
        $data['pdf_file_path'] = $magazine['pdf_file_path'];
        $data['file_size'] = $magazine['file_size'];
    }
    
    // Handle cover image upload
    if (isset($_FILES['cover_image']) && $_FILES['cover_image']['error'] === UPLOAD_ERR_OK) {
        $imageUpload = uploadImage($_FILES['cover_image']);
        if ($imageUpload['success']) {
            $data['cover_image_path'] = 'uploads/magazines/covers/' . $imageUpload['filename'];
            
            // Delete old cover if editing
            if ($editMode && !empty($magazine['cover_image_path'])) {
                deleteFile($magazine['cover_image_path']);
            }
        }
    } elseif ($editMode && !empty($magazine['cover_image_path'])) {
        $data['cover_image_path'] = $magazine['cover_image_path'];
    }
    
    // Create or update magazine
    if ($editMode) {
        if ($magazineController->updateMagazine($magazineId, $data)) {
            // Update categories
            $selectedCategories = $_POST['categories'] ?? [];
            $magazineController->assignCategories($magazineId, $selectedCategories);
            
            flash('flash_message', 'Magazine updated successfully!', 'success');
            redirect(base_url('admin/magazines.php'));
        } else {
            flash('flash_message', 'Error updating magazine.', 'danger');
        }
    } else {
        $magazineId = $magazineController->createMagazine($data);
        if ($magazineId) {
            // Assign categories
            $selectedCategories = $_POST['categories'] ?? [];
            $magazineController->assignCategories($magazineId, $selectedCategories);
            
            flash('flash_message', 'Magazine created successfully!', 'success');
            redirect(base_url('admin/magazines.php'));
        } else {
            flash('flash_message', 'Error creating magazine.', 'danger');
        }
    }
}

$pageTitle = ($editMode ? 'Edit' : 'Upload New') . ' Magazine - Admin';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= e($pageTitle) ?></title>
    <link rel="stylesheet" href="<?= base_url('assets/css/style.css') ?>">
    <style>
        .admin-container { max-width: 900px; margin: 40px auto; padding: 20px; }
        .admin-nav { background: #f8f9fa; padding: 15px; border-radius: 8px; margin-bottom: 30px; }
        .admin-nav a { margin-right: 15px; padding: 8px 15px; background: #007bff; color: white; text-decoration: none; border-radius: 5px; display: inline-block; margin-bottom: 5px; }
        .admin-nav a:hover { background: #0056b3; }
        .admin-nav a.active { background: #28a745; }
        .form-group { margin-bottom: 20px; }
        .form-group label { display: block; font-weight: bold; margin-bottom: 5px; }
        .form-group input, .form-group textarea, .form-group select { width: 100%; padding: 10px; border: 1px solid #ddd; border-radius: 5px; font-size: 14px; }
        .form-group textarea { min-height: 100px; }
        .form-group small { display: block; color: #666; margin-top: 5px; }
        .checkbox-group { margin-bottom: 15px; }
        .checkbox-group label { display: inline-block; margin-right: 20px; }
        .checkbox-group input { width: auto; margin-right: 5px; }
        .btn { padding: 12px 24px; background: #007bff; color: white; border: none; border-radius: 5px; cursor: pointer; font-size: 16px; text-decoration: none; display: inline-block; }
        .btn:hover { background: #0056b3; }
        .btn-secondary { background: #6c757d; }
        .btn-secondary:hover { background: #5a6268; }
        .alert { padding: 15px; border-radius: 5px; margin-bottom: 20px; }
        .alert-success { background: #d4edda; color: #155724; }
        .alert-danger { background: #f8d7da; color: #721c24; }
        .category-checkboxes { display: flex; flex-wrap: wrap; gap: 10px; }
        .category-checkboxes label { flex: 0 0 calc(33.333% - 10px); }
    </style>
</head>
<body>
    <div class="admin-container">
        <h1><?= $editMode ? 'Edit' : 'Upload New' ?> Magazine</h1>
        
        <?php include 'partials/admin-nav.php'; ?>
        
        <?php flash('flash_message'); ?>
        
        <form method="POST" enctype="multipart/form-data">
            <?= csrf_field() ?>
            <?php if ($editMode): ?>
                <input type="hidden" name="magazine_id" value="<?= e($magazine['id']) ?>">
            <?php endif; ?>
            
            <div class="form-group">
                <label for="title">Magazine Title *</label>
                <input type="text" id="title" name="title" required
                       value="<?= $editMode ? e($magazine['title']) : '' ?>">
            </div>
            
            <div class="form-group">
                <label for="description">Description</label>
                <textarea id="description" name="description"><?= $editMode ? e($magazine['description']) : '' ?></textarea>
            </div>
            
            <div class="form-group">
                <label for="issue_number">Issue Number</label>
                <input type="text" id="issue_number" name="issue_number"
                       value="<?= $editMode ? e($magazine['issue_number']) : '' ?>"
                       placeholder="e.g., #42">
            </div>
            
            <div class="form-group">
                <label for="year">Year *</label>
                <input type="number" id="year" name="year" required
                       value="<?= $editMode ? e($magazine['year']) : date('Y') ?>"
                       min="2000" max="2100">
            </div>
            
            <div class="form-group">
                <label for="month">Month</label>
                <select id="month" name="month">
                    <option value="">Not specified</option>
                    <?php for ($m = 1; $m <= 12; $m++): ?>
                        <option value="<?= $m ?>" <?= ($editMode && $magazine['month'] == $m) ? 'selected' : '' ?>>
                            <?= date('F', mktime(0, 0, 0, $m, 1)) ?>
                        </option>
                    <?php endfor; ?>
                </select>
            </div>
            
            <div class="form-group">
                <label for="publication_date">Publication Date *</label>
                <input type="date" id="publication_date" name="publication_date"
                       value="<?= $editMode ? e($magazine['publication_date']) : date('Y-m-d') ?>">
            </div>
            
            <div class="form-group">
                <label for="pdf_file">PDF File <?= $editMode ? '(leave empty to keep current file)' : '*' ?></label>
                <input type="file" id="pdf_file" name="pdf_file" accept=".pdf" <?= $editMode ? '' : 'required' ?>>
                <?php if ($editMode && !empty($magazine['pdf_file_path'])): ?>
                    <small>Current: <?= e(basename($magazine['pdf_file_path'])) ?> 
                           (<?= formatFileSize($magazine['file_size']) ?>)</small>
                <?php endif; ?>
            </div>
            
            <div class="form-group">
                <label for="cover_image">Cover Image (optional)</label>
                <input type="file" id="cover_image" name="cover_image" accept="image/*">
                <?php if ($editMode && !empty($magazine['cover_image_path'])): ?>
                    <small>Current: <?= e(basename($magazine['cover_image_path'])) ?></small>
                <?php endif; ?>
            </div>
            
            <div class="form-group">
                <label>Categories</label>
                <div class="category-checkboxes">
                    <?php 
                    $selectedCategoryIds = $editMode ? array_column($magazine['categories'], 'id') : [];
                    foreach ($categories as $cat): 
                    ?>
                        <label>
                            <input type="checkbox" name="categories[]" value="<?= e($cat['id']) ?>"
                                   <?= in_array($cat['id'], $selectedCategoryIds) ? 'checked' : '' ?>>
                            <?= e($cat['category_name']) ?>
                        </label>
                    <?php endforeach; ?>
                </div>
            </div>
            
            <div class="checkbox-group">
                <label>
                    <input type="checkbox" name="is_public" value="1" 
                           <?= ($editMode && $magazine['is_public']) || !$editMode ? 'checked' : '' ?>>
                    Public (visible to all users)
                </label>
                
                <label>
                    <input type="checkbox" name="is_pro_only" value="1"
                           <?= ($editMode && $magazine['is_pro_only']) ? 'checked' : '' ?>>
                    PRO Only (requires PRO subscription to download)
                </label>
            </div>
            
            <button type="submit" class="btn"><?= $editMode ? 'Update' : 'Upload' ?> Magazine</button>
            <a href="<?= base_url('admin/magazines.php') ?>" class="btn btn-secondary">Cancel</a>
        </form>
    </div>
</body>
</html>
