<?php
require_once __DIR__ . '/../../includes/bootstrap.php';

requireAdmin();

$newsletterController = new NewsletterController(getPDO());

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    verify_csrf();
    
    $action = $_POST['action'] ?? '';
    
    if ($action === 'add') {
        $email = $_POST['email'] ?? '';
        if (filter_var($email, FILTER_VALIDATE_EMAIL)) {
            if ($newsletterController->addSubscriber($email, false)) {
                // Manually confirm them since admin is adding
                $subscriber = $newsletterController->getSubscriberByEmail($email);
                $pdo = getPDO();
                $stmt = $pdo->prepare("UPDATE newsletter_subscribers SET is_confirmed = 1, confirmed_at = NOW() WHERE id = :id");
                $stmt->execute(['id' => $subscriber['id']]);
                
                flash('flash_message', 'Subscriber added successfully!', 'success');
            } else {
                flash('flash_message', 'Subscriber already exists or error occurred.', 'danger');
            }
        } else {
            flash('flash_message', 'Invalid email address.', 'danger');
        }
    } elseif ($action === 'delete') {
        $id = $_POST['id'] ?? null;
        if ($newsletterController->deleteSubscriber($id)) {
            flash('flash_message', 'Subscriber deleted successfully!', 'success');
        }
    } elseif ($action === 'export') {
        $csv = $newsletterController->exportToCSV(true);
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="newsletter_subscribers_' . date('Y-m-d') . '.csv"');
        echo $csv;
        exit;
    }
    
    redirect(base_url('admin/newsletter.php'));
}

// Get filter
$filter = $_GET['filter'] ?? 'confirmed';
$filters = [];
if ($filter === 'confirmed') {
    $filters['is_confirmed'] = 1;
} elseif ($filter === 'pending') {
    $filters['is_confirmed'] = 0;
}

$subscribers = $newsletterController->getAllSubscribers($filters);
$stats = $newsletterController->getStatistics();

$pageTitle = 'Newsletter Subscribers - Admin';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= e($pageTitle) ?></title>
    <link rel="stylesheet" href="<?= base_url('assets/css/style.css') ?>">
    <style>
        .admin-container { max-width: 1400px; margin: 40px auto; padding: 20px; }
        .admin-nav { background: #f8f9fa; padding: 15px; border-radius: 8px; margin-bottom: 30px; }
        .admin-nav a { margin-right: 15px; padding: 8px 15px; background: #007bff; color: white; text-decoration: none; border-radius: 5px; display: inline-block; margin-bottom: 5px; }
        .admin-nav a:hover { background: #0056b3; }
        .admin-nav a.active { background: #28a745; }
        .stats-grid { display: grid; grid-template-columns: repeat(4, 1fr); gap: 20px; margin-bottom: 30px; }
        .stat-card { background: #fff; border: 1px solid #ddd; border-radius: 8px; padding: 20px; text-align: center; }
        .stat-card h3 { margin: 0; font-size: 32px; color: #007bff; }
        .stat-card p { margin: 5px 0 0; color: #666; }
        .btn { padding: 10px 20px; background: #007bff; color: white; border: none; border-radius: 5px; cursor: pointer; text-decoration: none; display: inline-block; }
        .btn:hover { background: #0056b3; }
        .btn-danger { background: #dc3545; }
        .btn-danger:hover { background: #c82333; }
        .btn-success { background: #28a745; }
        .btn-success:hover { background: #218838; }
        .btn-sm { padding: 5px 10px; font-size: 12px; }
        table { width: 100%; border-collapse: collapse; margin-top: 20px; }
        th, td { padding: 12px; text-align: left; border-bottom: 1px solid #ddd; }
        th { background: #f8f9fa; font-weight: bold; }
        .alert { padding: 15px; border-radius: 5px; margin-bottom: 20px; }
        .alert-success { background: #d4edda; color: #155724; }
        .alert-danger { background: #f8d7da; color: #721c24; }
        .badge { padding: 4px 8px; border-radius: 3px; font-size: 11px; font-weight: bold; }
        .badge-confirmed { background: #28a745; color: white; }
        .badge-pending { background: #ffc107; color: #333; }
        .filter-tabs { margin-bottom: 20px; }
        .filter-tabs a { padding: 10px 20px; background: #f8f9fa; border: 1px solid #ddd; text-decoration: none; color: #333; display: inline-block; margin-right: 5px; border-radius: 5px 5px 0 0; }
        .filter-tabs a.active { background: #007bff; color: white; border-color: #007bff; }
        .modal { display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 1000; }
        .modal.active { display: flex; align-items: center; justify-content: center; }
        .modal-content { background: white; padding: 30px; border-radius: 10px; max-width: 500px; width: 90%; }
        .form-group { margin-bottom: 15px; }
        .form-group label { display: block; font-weight: bold; margin-bottom: 5px; }
        .form-group input { width: 100%; padding: 10px; border: 1px solid #ddd; border-radius: 5px; }
    </style>
</head>
<body>
    <div class="admin-container">
        <h1>Admin Panel - Newsletter Subscribers</h1>
        
        <?php include 'partials/admin-nav.php'; ?>
        
        <?php flash('flash_message'); ?>
        
        <div class="stats-grid">
            <div class="stat-card">
                <h3><?= e($stats['total']) ?></h3>
                <p>Total Subscribers</p>
            </div>
            <div class="stat-card">
                <h3><?= e($stats['confirmed']) ?></h3>
                <p>Confirmed</p>
            </div>
            <div class="stat-card">
                <h3><?= e($stats['pending']) ?></h3>
                <p>Pending</p>
            </div>
            <div class="stat-card">
                <h3><?= e($stats['unsubscribed']) ?></h3>
                <p>Unsubscribed</p>
            </div>
        </div>
        
        <div style="margin-bottom: 20px;">
            <button class="btn" onclick="openModal()">+ Add Subscriber</button>
            <form method="POST" style="display: inline;">
                <?= csrf_field() ?>
                <input type="hidden" name="action" value="export">
                <button type="submit" class="btn btn-success">Export to CSV</button>
            </form>
        </div>
        
        <div class="filter-tabs">
            <a href="?filter=all" class="<?= $filter === 'all' ? 'active' : '' ?>">All</a>
            <a href="?filter=confirmed" class="<?= $filter === 'confirmed' ? 'active' : '' ?>">Confirmed</a>
            <a href="?filter=pending" class="<?= $filter === 'pending' ? 'active' : '' ?>">Pending</a>
        </div>
        
        <table>
            <thead>
                <tr>
                    <th>ID</th>
                    <th>Email</th>
                    <th>Status</th>
                    <th>Subscribe Date</th>
                    <th>Confirmed Date</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($subscribers)): ?>
                    <tr>
                        <td colspan="6" style="text-align: center; padding: 40px;">
                            No subscribers found.
                        </td>
                    </tr>
                <?php else: ?>
                    <?php foreach ($subscribers as $sub): ?>
                    <tr>
                        <td><?= e($sub['id']) ?></td>
                        <td><?= e($sub['email']) ?></td>
                        <td>
                            <?php if ($sub['is_confirmed']): ?>
                                <span class="badge badge-confirmed">Confirmed</span>
                            <?php else: ?>
                                <span class="badge badge-pending">Pending</span>
                            <?php endif; ?>
                        </td>
                        <td><?= e(date('Y-m-d H:i', strtotime($sub['created_at']))) ?></td>
                        <td><?= $sub['confirmed_at'] ? e(date('Y-m-d H:i', strtotime($sub['confirmed_at']))) : '-' ?></td>
                        <td>
                            <form method="POST" style="display:inline;" onsubmit="return confirm('Delete this subscriber?');">
                                <?= csrf_field() ?>
                                <input type="hidden" name="action" value="delete">
                                <input type="hidden" name="id" value="<?= e($sub['id']) ?>">
                                <button type="submit" class="btn btn-sm btn-danger">Delete</button>
                            </form>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
    
    <!-- Add Subscriber Modal -->
    <div id="addModal" class="modal">
        <div class="modal-content">
            <h2>Add Subscriber</h2>
            <form method="POST">
                <?= csrf_field() ?>
                <input type="hidden" name="action" value="add">
                
                <div class="form-group">
                    <label for="email">Email Address</label>
                    <input type="email" id="email" name="email" required>
                </div>
                
                <button type="submit" class="btn">Add Subscriber</button>
                <button type="button" class="btn btn-danger" onclick="closeModal()">Cancel</button>
            </form>
        </div>
    </div>
    
    <script>
        function openModal() {
            document.getElementById('addModal').classList.add('active');
        }
        function closeModal() {
            document.getElementById('addModal').classList.remove('active');
        }
    </script>
</body>
</html>
