<?php

// Minimal, isolated smoke tests for auth + models.
// Safety: requires DB_DSN to be a SQLite DSN (e.g., sqlite::memory:) so we don't touch real data.

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

$dsn = getenv('DB_DSN') ?: '';
if ($dsn === '' || stripos($dsn, 'sqlite:') !== 0) {
    echo "Set DB_DSN=sqlite::memory: (or a test sqlite file) to run these tests without touching prod data.\n";
    exit(1);
}

$drivers = PDO::getAvailableDrivers();
if (!in_array('sqlite', $drivers, true)) {
    echo "PDO SQLite driver is not installed. Enable pdo_sqlite or use a PHP build with SQLite support.\n";
    exit(1);
}

$pdo = new PDO($dsn, null, null, [
    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    PDO::ATTR_EMULATE_PREPARES => false,
]);

// SQLite-friendly schema
$pdo->exec("
    CREATE TABLE users (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        email TEXT NOT NULL UNIQUE,
        password_hash TEXT NOT NULL,
        name TEXT,
        role TEXT DEFAULT 'user',
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
    );

    CREATE TABLE user_stats (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        user_id INTEGER NOT NULL UNIQUE,
        total_eye_exercises_minutes INTEGER DEFAULT 0,
        total_focus_minutes INTEGER DEFAULT 0,
        total_breathing_sessions INTEGER DEFAULT 0,
        last_activity_at DATETIME,
        FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
    );

    CREATE TABLE articles (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        slug TEXT NOT NULL UNIQUE,
        title TEXT NOT NULL,
        excerpt TEXT,
        body TEXT NOT NULL,
        category TEXT,
        published INTEGER DEFAULT 1,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
    );
");

require_once __DIR__ . '/../includes/models/UserModel.php';
require_once __DIR__ . '/../includes/models/StatsModel.php';
require_once __DIR__ . '/../includes/models/ArticleModel.php';
require_once __DIR__ . '/../includes/controllers/AuthController.php';

$userModel = new UserModel($pdo);
$statsModel = new StatsModel($pdo);
$articleModel = new ArticleModel($pdo);
$authController = new AuthController($pdo);

$results = [];

// Register user
$register = $authController->register('tester@example.com', 'secret123', 'Test User');
$results[] = ['Register user', empty($register['error'])];

// Login success
$loginOk = $authController->login('tester@example.com', 'secret123');
$results[] = ['Login with correct password', $loginOk === true];

// Login failure
$loginFail = $authController->login('tester@example.com', 'wrongpass');
$results[] = ['Login with bad password fails', $loginFail === false];

// Stats increments
$user = $userModel->findByEmail('tester@example.com');
$statsModel->incrementFocusMinutes($user['id'], 15);
$statsModel->incrementBreathingSessions($user['id']);
$stats = $statsModel->getUserStats($user['id']);
$results[] = ['Stats focus minutes updated', $stats['total_focus_minutes'] === 15];
$results[] = ['Stats breathing sessions updated', $stats['total_breathing_sessions'] === 1];

// Article creation and fetch
$articleModel->createArticle('Test Title', 'test-title', 'Body here', 'Excerpt', 'Wellness', 1);
$fetched = $articleModel->getArticleBySlug('test-title');
$results[] = ['Article retrievable by slug', $fetched !== false && $fetched['title'] === 'Test Title'];

echo "Model/Auth smoke tests\n";
echo "----------------------\n";
foreach ($results as [$label, $ok]) {
    echo sprintf("%-35s : %s\n", $label, $ok ? 'OK' : 'FAIL');
}

// Show summary
$failures = array_filter($results, fn($r) => !$r[1]);
echo "\n" . count($failures) . " failures out of " . count($results) . " checks.\n";
exit(empty($failures) ? 0 : 1);
